
/*******************************************************************************
 * ON Semiconductor                                                            *
 * NCV7381B Evaluation Board V1                                                *
 *                                                                             *
 * FlexRay_BD.c                                                                *
 * FlexRay Bus Driver handling functions                                       *
 *                                                                             *
 * Created by: Filip Brtan                                                     *
 * Date: 20.06.2019                                                            *
 *******************************************************************************/


/** S12X derivative information */ 
#include "mc9s12xf512.h"       

/** Macro definitions for general purpose I/O handling  */
#include "GPIO_macros.h"

/** Variable types and common definitions */
#include "typedefs.h" 

/** UNIFIED driver implementation */         
#include "Fr_UNIFIED.h" 

/** FlexRay Bus Driver functions definitions */
#include "FlexRay_BD.h"
				

				
/*******************************************************************************/
/**
* \brief    100ns wait function
* \author   Filip Brtan
* \param    count100ns: number of 100ns wait cycles
* \return   void
*/				
void wait_100ns(UINT16 count100ns) {
  PITLD0 = count100ns;
  PITTF = 1;
  PITCFLMT_PFLMT0 = 1;
  PITFLT_PITFLT0 = 1;
  
  while(!PITTF_PITTF0);
}

/*******************************************************************************/
/**
* \brief    1us wait function
* \author   Filip Brtan
* \param    count1us: number of 100ns wait cycles
* \return   void
*/
void wait_1us(UINT16 count1us) {
  PITLD0 = (word)((word)10 * count1us);
  PITTF = 1;
  PITCFLMT_PFLMT0 = 1;
  PITFLT_PITFLT0 = 1;
  
  while(!PITTF_PITTF0);
}

// wait 1ms
void wait_1ms(void) {
  PITLD0 = 9999;
  PITTF = 1;
  PITCFLMT_PFLMT0 = 1;
  PITFLT_PITFLT0 = 1;
  
  while(!PITTF_PITTF0);
}


/*******************************************************************************/
/**
 * \brief    Status Register readout
 * \author   Filip Brtan
 * \param    channel: Define channel from which to read the status 
 * \return   FlexRay Bus Driver Status Register content
 */
Fr_bd_status_register fr_bd_read_status(Fr_channel_type channel) {
  Fr_bd_status_register Status;
  byte i;
  bool FR_EN_A_before;
  bool FR_EN_B_before;
  
  switch (channel) {
  
    case FR_CHANNEL_A: 
      FR_EN_A_before = FR_EN_A;

      FR_EN_A = 1;
      wait_100ns(39);
      FR_EN_A = 0;
      wait_100ns(39);
        
      for (i=0; i<16; i++) {
        FR_EN_A = 1;
        wait_100ns(39);
        FR_EN_A = 0;
        wait_100ns(34);
        Status.Flags.Word >>= 1;
        Status.Flags.Bits.S15 = FR_ERRN_A;
            
      }
      for (i=0; i<16; i++) {
        FR_EN_A = 1;
        wait_100ns(38);
        FR_EN_A = 0;
        wait_100ns(34);
        Status.Version.Word >>= 1;
        Status.Version.Bits.S31 = FR_ERRN_A;    
      } 
      FR_EN_A = FR_EN_A_before;
      break;
      
    case FR_CHANNEL_B: 
      
      FR_EN_B_before = FR_EN_B;
      //PTT_PTT0 = FR_EN_B;

      FR_EN_B = 1;
      wait_100ns(39);
      FR_EN_B = 0;
      wait_100ns(39);
        
      for (i=0; i<16; i++) {
        FR_EN_B = 1;
        wait_100ns(38);
        FR_EN_B = 0;
        wait_100ns(34);
        Status.Flags.Word >>= 1;
        Status.Flags.Bits.S15 = FR_ERRN_B;
            
      }
      for (i=0; i<16; i++) {
        FR_EN_B = 1;
        wait_100ns(38);
        FR_EN_B = 0;
        wait_100ns(34);
        Status.Version.Word >>= 1;
        Status.Version.Bits.S31 = FR_ERRN_B;    
      } 
      //FR_EN_B = PTT_PTT0;
      FR_EN_B = FR_EN_B_before;
      break;
      
    default: 
      Status.Flags.Word = 0;
      Status.Version.Word = 0; 
      break;
  }
  
  return Status; 
}

/*******************************************************************************/
/**
 * \brief    Channel A Status Register readout
 * \author   Filip Brtan
 * \param    void 
 * \return   FlexRay Bus Driver A Status Register content
 */
Fr_bd_status_register fr_bd_read_status_A(void) {
  Fr_bd_status_register Status;
  byte i;
  bool FR_EN_A_before;
  FR_EN_A_before = FR_EN_A;

  FR_EN_A = 1;
  wait_100ns(39);
  FR_EN_A = 0;
  wait_100ns(39);
    
  for (i=0; i<16; i++) {
    FR_EN_A = 1;
    wait_100ns(39);
    FR_EN_A = 0;
    wait_100ns(34);
    Status.Flags.Word >>= 1;
    Status.Flags.Bits.S15 = FR_ERRN_A;
        
  }
  for (i=0; i<16; i++) {
    FR_EN_A = 1;
    wait_100ns(38);
    FR_EN_A = 0;
    wait_100ns(34);
    Status.Version.Word >>= 1;
    Status.Version.Bits.S31 = FR_ERRN_A;    
  } 
  FR_EN_A = FR_EN_A_before;
  return Status; 
}


/*******************************************************************************/
/**
 * \brief    Channel B Status Register readout
 * \author   Filip Brtan
 * \param    void 
 * \return   FlexRay Bus Driver B Status Register content
 */
Fr_bd_status_register fr_bd_read_status_B(void) {
  Fr_bd_status_register Status;
  byte i;
  bool FR_EN_B_before;
  FR_EN_B_before = FR_EN_B;
  //PTT_PTT0 = FR_EN_B;

  FR_EN_B = 1;
  wait_100ns(39);
  FR_EN_B = 0;
  wait_100ns(39);
    
  for (i=0; i<16; i++) {
    FR_EN_B = 1;
    wait_100ns(38);
    FR_EN_B = 0;
    wait_100ns(34);
    Status.Flags.Word >>= 1;
    Status.Flags.Bits.S15 = FR_ERRN_B;
        
  }
  for (i=0; i<16; i++) {
    FR_EN_B = 1;
    wait_100ns(38);
    FR_EN_B = 0;
    wait_100ns(34);
    Status.Version.Word >>= 1;
    Status.Version.Bits.S31 = FR_ERRN_B;    
  } 
  //FR_EN_B = PTT_PTT0;
  FR_EN_B = FR_EN_B_before;
  return Status; 
}

/* Channel A+B Status Register readout */
void fr_bd_read_status_AB(Fr_bd_status_register *status1, Fr_bd_status_register *status2) {
  Fr_bd_status_register Status1;
  Fr_bd_status_register Status2;
  byte i;
  bool FR_EN_A_before;
  bool FR_EN_B_before;
  FR_EN_A_before = FR_EN_A;
  FR_EN_B_before = FR_EN_B;

  FR_EN_A = 1;
  FR_EN_B = 1;
  wait_100ns(39);
  FR_EN_A = 0;
  FR_EN_B = 0;
  wait_100ns(38);
    
  for (i=0; i<16; i++) {
    FR_EN_A = 1;
    FR_EN_B = 1;
    wait_100ns(38);
    FR_EN_A = 0;
    FR_EN_B = 0;
    wait_100ns(28);
    Status1.Flags.Word >>= 1;
    Status2.Flags.Word >>= 1;
    Status1.Flags.Bits.S15 = FR_ERRN_A;
    Status2.Flags.Bits.S15 = FR_ERRN_B;
        
  }
  for (i=0; i<16; i++) {
    FR_EN_A = 1;
    FR_EN_B = 1;
    wait_100ns(38);
    FR_EN_A = 0;
    FR_EN_B = 0;
    wait_100ns(28);
    Status1.Version.Word >>= 1;
    Status2.Version.Word >>= 1;
    Status1.Version.Bits.S31 = FR_ERRN_A; 
    Status2.Version.Bits.S31 = FR_ERRN_B;    
  } 

  FR_EN_A = FR_EN_A_before;
  FR_EN_B = FR_EN_B_before;
  
  *status1 = Status1 ;
  *status2 = Status2;
}


/*  */
void fr_bd_change_mode(Fr_channel_type channel, Fr_BD_mode_type mode)               
{    
    switch(channel) {
      case FR_CHANNEL_A: 
        switch(mode) {
          case FR_BD_NORMAL : 
            FR_EN_A = 1;
            FR_STBN_A = 1;
            break;
          case FR_BD_RECEIVE_ONLY : 
            FR_EN_A = 0;
            FR_STBN_A = 1;
            break;      
          case FR_BD_STANDBY : 
            FR_EN_A = 0;
            FR_STBN_A = 0;
            break;
          case FR_BD_GOTOSLEEP : 
          case FR_BD_SLEEP :
            FR_EN_A = 1;
            FR_STBN_A = 0;
            break;
          default: break;
        }break;
      case FR_CHANNEL_B: 
        switch(mode) {
          case FR_BD_NORMAL : 
            FR_EN_B = 1;
            FR_STBN_B = 1;
            break;
          case FR_BD_RECEIVE_ONLY : 
            FR_EN_B = 0;
            FR_STBN_B = 1;
            break;      
          case FR_BD_STANDBY : 
            FR_EN_B = 0;
            FR_STBN_B = 0;
            break;
          case FR_BD_GOTOSLEEP : 
          case FR_BD_SLEEP :
            FR_EN_B = 1;
            FR_STBN_B = 0;
            break;
          default: break;
        }break;
      case FR_CHANNEL_AB: 
        switch(mode) {
          case FR_BD_NORMAL : 
          
            FR_EN_A = 1;
            FR_STBN_A = 1;
            FR_EN_B = 1;
            FR_STBN_B = 1;
            break;
          case FR_BD_RECEIVE_ONLY : 
            FR_EN_A = 0;
            FR_STBN_A = 1;
            FR_EN_B = 0;
            FR_STBN_B = 1;
            break;      
          case FR_BD_STANDBY : 
            FR_EN_A = 0;
            FR_EN_B = 0;
            FR_STBN_A = 0;
            FR_STBN_B = 0;
            break;
          case FR_BD_GOTOSLEEP :
          case FR_BD_SLEEP : 
            FR_EN_A = 1;
            FR_STBN_A = 0;
            FR_EN_B = 1;
            FR_STBN_B = 0;
            break;
          default: break;
        }break;
      default: break;
    }
}

/* Application Test Test Patterns */

/* Valid Remote Wakeup Test Patterns */
void TestPatternRemoteWakeup(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100]
      // 18us
      NOP();               // [25]
      i=238; while(--i);   // [15875] 25ns*(1+3*i)
      FR_CLR_TXD_TXEN_A(); // [100]  
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 
      // 18us
      NOP();               // [25]
      i=238; while(--i);   // [15875] 25ns*(1+3*i)
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100] 
      // 18us
      NOP();               // [25]
      i=238; while(--i);   // [15875] 25ns*(1+3*i)
      FR_CLR_TXD_TXEN_B(); // [100]  
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100]
      // 18us
      NOP();               // [25]
      i=238; while(--i);   // [15875] 25ns*(1+3*i)
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
      // 18us
      NOP();                // [25]
      i=238; while(--i);    // [15875] 25ns*(1+3*i)
      FR_CLR_TXD_TXEN_AB(); // [100]   
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
      // 18us
      NOP();                // [25]
      i=238; while(--i);    // [15875] 25ns*(1+3*i)
  }
}

/* Alternative Remote Wakeup Test Patterns */
void TestPatternRemoteWakeupAlternative(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXD_A = 1;        // [100]
      // 18us
      NOP();               // [25]
      i=238; while(--i);   // [15875] 25ns*(1+3*i)
      FR_TXD_A = 0;        // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXD_B = 1;        // [100]
      // 18us
      NOP();               // [25]
      i=238; while(--i);   // [15875] 25ns*(1+3*i)
      FR_TXD_B = 0;        // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100]
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXD_AB();      // [100]
      // 18us
      NOP();                // [25]
      i=238; while(--i);    // [15875] 25ns*(1+3*i)
      FR_CLR_TXD_AB();      // [100]   
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
  }
}


/* Shorted Remote Wakeup Test Patterns */
void TestPatternRemoteWakeupShorted(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100]
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_CLR_TXD_TXEN_A(); // [100]  
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100] 
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_CLR_TXD_TXEN_B(); // [100]  
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100]
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_CLR_TXD_TXEN_AB(); // [100]   
      // 4.1us
      i=53; while(--i);    // [4000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
  }
}

/* Remote Wakeup Frame Payload Test Patterns */
void TestPatternRemoteWakeupFrame(Fr_channel_type channel) {
  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A();  
                             
        asm LDAB  #5         // [25]
      asm START1A:  
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START1A  // [75]
        
        asm LDAB  #5         // [25]
      asm START2A:  
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START2A  // [75]
      
        asm LDAB  #5         // [25]
      asm START3A:  
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START3A  // [75]
        
        asm LDAB  #5         // [25]
      asm START4A:  
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START4A  // [75]
      
        asm LDAB  #5         // [25]
      asm START5A:  
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START5A  // [75]
        
        asm LDAB  #5         // [25]
      asm START6A:  
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 0;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START6A  // [75]    
      
        asm LDAB  #6         // [25]
      asm START7A:  
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 0;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100]
        FR_TXD_A = 1;        // [100] 
        FR_TXD_A = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START7A  // [75]     

      FR_SET_TXD_TXEN_A(); //   [100] 
  } else if (channel == FR_CHANNEL_B) {
            //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_B();  
                             
        asm LDAB  #5         // [25]
      asm START1B:  
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START1B  // [75]
        
        asm LDAB  #5         // [25]
      asm START2B:  
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START2B  // [75]
      
        asm LDAB  #5         // [25]
      asm START3B:  
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START3B  // [75]
        
        asm LDAB  #5         // [25]
      asm START4B:  
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START4B  // [75]
      
        asm LDAB  #5         // [25]
      asm START5B:  
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START5B  // [75]
        
        asm LDAB  #5         // [25]
      asm START6B:  
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 0;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START6B  // [75]    
      
        asm LDAB  #6         // [25]
      asm START7B:  
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 0;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100]
        FR_TXD_B = 1;        // [100] 
        FR_TXD_B = 1;        // [100] 
        asm NOP              // [25]
        asm DBNE  B,START7B  // [75]     

      FR_SET_TXD_TXEN_B(); //   [100]
  }else if (channel == FR_CHANNEL_AB)  {  
      
  }
}


/* Non Valid Remote Wakeup Test Patterns - short IDLE*/
void TestPatternRemoteWakeupFailShortIdle(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_A = 1;       // [100]
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_TXEN_A = 0;       // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_B = 1;       // [100]
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_TXEN_B = 0;       // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100]
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXEN_AB();     // [100]
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_CLR_TXEN_AB();     // [100]   
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
  }
}

/* Non Valid Remote Wakeup Test Patterns - long IDLE */
void TestPatternRemoteWakeupFailLongIdle(Fr_channel_type channel) {
  byte i;
  word j;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_A = 1;       // [100]
      // 140us
      NOP(); NOP();        // [50]
      j=1864; while(--j);  // [139850] 25ns*(2+3*i)
      FR_TXEN_A = 0;       // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_B = 1;       // [100]
      // 140us
      NOP(); NOP();        // [50]
      j=1864; while(--j);  // [139850] 25ns*(2+3*i)
      FR_TXEN_B = 0;       // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100] 
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXEN_AB();     // [100]
      // 140us
      NOP(); NOP();        // [50]
      j=1864; while(--j);  // [139850] 25ns*(2+3*i)
      FR_CLR_TXEN_AB();     // [100]   
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
  }
}

/* Non Valid Remote Wakeup Test Patterns - short firs LOW */
void TestPatternRemoteWakeupFailShortLow1(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_TXEN_A = 1;       // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_A = 0;       // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_TXEN_B = 1;       // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_B = 0;       // [100] 
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100]
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_SET_TXEN_AB();     // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_CLR_TXEN_AB();     // [100]   
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
  }
}

/* Non Valid Remote Wakeup Test Patterns - short second LOW */
void TestPatternRemoteWakeupFailShortLow2(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_A = 1;       // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_A = 0;       // [100] 
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_B = 1;       // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_TXEN_B = 0;       // [100] 
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100]
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 6us
      NOP();                // [25]
      i=78; while(--i);     // [5875] 25ns*(1+3*i)
      FR_SET_TXEN_AB();     // [100]
      // 6us
      NOP();               // [25]
      i=78; while(--i);    // [5875] 25ns*(1+3*i)
      FR_CLR_TXEN_AB();     // [100]   
      // 0.9us
      NOP();               // [25]
      i=10; while(--i);    // [775] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100]
  }
}

/* TSS Test Patterns */
void TestPatternTSS(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]
      // 11 gdBit
      i=13; while(--i);    // [1000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_A(); // [100] 

  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]
      // 11 gdBit
      i=13; while(--i);    // [1000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_B(); // [100] 
  }else if (channel == FR_CHANNEL_AB)  { 
      FR_CLR_TXD_TXEN_AB(); // [100]
      // 11 gdBit
      i=13; while(--i);    // [1000] 25ns*(1+3*i)
      FR_SET_TXD_TXEN_AB(); // [100] 
  }
}


/* TSS + one 10gdBit High Test Pattern */
void TestPatternTSS_HIGH(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_TXD_A = 1;        //   [100] 
      // 10 gdBit HIGH     // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_SET_TXD_TXEN_A(); //   [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_TXD_B = 1;        //   [100] 
      // 10 gdBit HIGH     // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_SET_TXD_TXEN_B(); //   [100]
  }else if (channel == FR_CHANNEL_AB)  { 
      FR_CLR_TXD_TXEN_AB(); 
      // 11 gdBit TSS
      i=13; while(--i);     // [1000] 25ns*(1+3*i)
      FR_SET_TXD_AB();      // [100] 
      // 10 gdBit HIGH
      i=11; while(--i);     // [850] 25ns*(1+3*i)
      NOP(); NOP();         // [50]
      FR_SET_TXD_TXEN_AB(); // [100] 
  }
}

/* TSS + one 10gdBit LOW Test Pattern */
void TestPatternTSS_LOW(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); 
      // 11 gdBit TSS + 10 gdBit LOW // [2100]=
      i=26; while(--i);    //   [1975]+ 
      NOP();               //   [25]+
      FR_SET_TXD_TXEN_A(); //   [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); 
      // 11 gdBit TSS + 10 gdBit LOW // [2100]=
      i=26; while(--i);    //   [1975]+ 
      NOP();               //   [25]+
      FR_SET_TXD_TXEN_B(); //   [100] 
  }else if (channel == FR_CHANNEL_AB)  { 
      FR_CLR_TXD_TXEN_AB(); 
      // 11 gdBit TSS + 10 gdBit LOW // [2100]=
      i=26; while(--i);     //   [1975]+ 
      NOP();                //   [25]+
      FR_SET_TXD_TXEN_AB(); //   [100] 
  }
}

/* TSS + one 10gdBit High + one 10gdBit LOW Test Pattern */
void TestPatternTSS_HIGH_LOW(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_TXD_A = 1;        //   [100] 
      // 10 gdBit HIGH     // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_TXD_A = 0;        //   [100] 
      // 10 gdBit LOW      // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_SET_TXD_TXEN_A(); //   [100]  
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_TXD_B = 1;        //   [100] 
      // 10 gdBit HIGH     // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_TXD_B = 0;        //   [100] 
      // 10 gdBit LOW      // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_SET_TXD_TXEN_B(); //   [100]  
  }else if (channel == FR_CHANNEL_AB)  { 
      FR_CLR_TXD_TXEN_AB(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_SET_TXD_AB();     //   [100] 
      // 10 gdBit HIGH     // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_CLR_TXD_AB();     //   [100] 
      // 10 gdBit LOW      // [1000]=
      i=11; while(--i);    //   [850]+ 
      NOP(); NOP();        //   [50]+
      FR_SET_TXD_TXEN_AB(); //  [100]  
  }
}

/* TSS + one 50/50 Test Pattern */
void TestPatternTSS_50_50(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_TXD_A = 1;        //   [100] 
      // One 50/50 test pattern
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_SET_TXD_TXEN_A(); //   [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_TXD_B = 1;        //   [100] 
      // One 50/50 test pattern
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_SET_TXD_TXEN_B(); //   [100] 
  }else if (channel == FR_CHANNEL_AB)  { 
      FR_CLR_TXD_TXEN_AB(); 
      // 11 gdBit TSS      // [1100]=
      i=13; while(--i);    //   [1000]+ 
      FR_SET_TXD_AB();     //   [100] 
      // One 50/50 test pattern
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100]
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_TXEN_AB(); //  [100]  
  }
}

/* TSS + one 10/90 Test Pattern */
void TestPatternTSS_10_90(Fr_channel_type channel) {
  byte i;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); 
      // 11 gdBit TSS      //   [1100]=
      i=13; while(--i);    //     [1000]+ 
      FR_TXD_A = 1;        //     [100] 
      // One 10/90 test pattern [1000]=
      FR_TXD_A = 0;        //     [100]+
      i=10; while(--i);    //     [775]+
      NOP();               //     [25]+
      FR_SET_TXD_TXEN_A(); //     [100] 
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); 
      // 11 gdBit TSS      //   [1100]=
      i=13; while(--i);    //     [1000]+ 
      FR_TXD_B = 1;        //     [100] 
      // One 10/90 test pattern [1000]=
      FR_TXD_B = 0;        //     [100]+
      i=10; while(--i);    //     [775]+
      NOP();               //     [25]+
      FR_SET_TXD_TXEN_B(); //     [100] 
  }else if (channel == FR_CHANNEL_AB)  { 
      FR_CLR_TXD_TXEN_AB(); 
      // 11 gdBit TSS      //   [1100]=
      i=13; while(--i);    //     [1000]+ 
      FR_SET_TXD_AB();     //     [100] 
      // One 10/90 test pattern [1000]=
      FR_CLR_TXD_AB();     //     [100]+
      i=10; while(--i);    //     [775]+
      NOP();               //     [25]+
      FR_SET_TXD_TXEN_AB(); //    [100] 
  }
}

/* Two 50/50 Test Patterns */
void TestPattern_Two_50_50(Fr_channel_type channel) {
  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_TXD_A = 1;        //   [100]      
      FR_TXEN_A = 0;       //   [100]
      
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]
      FR_TXD_A = 1;        //   [100] 
      FR_TXD_A = 0;        //   [100]     
      FR_SET_TXD_TXEN_A(); //   [100]

  } else if (channel == FR_CHANNEL_B) {
      FR_TXD_B = 1;        //   [100]      
      FR_TXEN_B = 0;       //   [100]
      
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]
      FR_TXD_B = 1;        //   [100] 
      FR_TXD_B = 0;        //   [100]     
      FR_SET_TXD_TXEN_B(); //   [100]
  }else if (channel == FR_CHANNEL_AB)  { 
      FR_SET_TXD_AB();     //   [100]
      FR_CLR_TXEN_AB(); 
      
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100]
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100]
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_AB();     //   [100] 
      FR_CLR_TXD_AB();     //   [100]
      FR_SET_TXD_TXEN_AB(); //  [100]  
  }
}


/* Current Measurement Test Patterns - long IDLE */
void TestPatternCurrentMeasurement(Fr_channel_type channel) {
  byte i;
  word j;

  if (channel == FR_CHANNEL_A) {
      //                      [time in ns]
      //=========================================
      FR_CLR_TXD_TXEN_A(); // [100]    
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i)      
      FR_SET_TXD_TXEN_A(); // [100]
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i) 
      FR_CLR_TXD_TXEN_A(); // [100]     
      // 1us
      NOP();NOP();         // [50]
      i=11; while(--i);    // [850] 25ns*(1+3*i)     
      FR_TXD_A = 1;        // [100]
      // 649us
      NOP();               // [25]
      j=8651; while(--j);  // [648875] 25ns*(2+3*i) 
      FR_SET_TXD_TXEN_A(); // [100] 
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i)       
  } else if (channel == FR_CHANNEL_B) {
      FR_CLR_TXD_TXEN_B(); // [100]    
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i)      
      FR_SET_TXD_TXEN_B(); // [100]
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i) 
      FR_CLR_TXD_TXEN_B(); // [100]     
      // 1us
      NOP();NOP();         // [50]
      i=11; while(--i);    // [850] 25ns*(1+3*i)     
      FR_TXD_B = 1;        // [100]
      // 649us
      NOP();               // [25]
      j=8651; while(--j);  // [648875] 25ns*(2+3*i) 
      FR_SET_TXD_TXEN_B(); // [100] 
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i)  
  }else if (channel == FR_CHANNEL_AB)  {  
      FR_CLR_TXD_TXEN_AB(); // [100]    
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i)      
      FR_SET_TXD_TXEN_AB();// [100]
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i) 
      FR_CLR_TXD_TXEN_AB();// [100]     
      // 1us
      NOP();NOP();         // [50]
      i=11; while(--i);    // [850] 25ns*(1+3*i)     
      FR_SET_TXD_AB();     // [100]
      // 649us
      NOP();               // [25]
      j=8651; while(--j);  // [648875] 25ns*(2+3*i) 
      FR_SET_TXD_TXEN_AB();// [100] 
      // 650us
      NOP();NOP();         // [50]
      j=8664; while(--j);  // [649850] 25ns*(2+3*i)  
  }
}



/* Configuration of CPU pins connected to FR bus driver */
void Fr_bd_ports_init(void) {
 
  /*          I  BSER   */
  /*          N  GTRX   */
  /*          H  EBRE   */
  /*          V   NNN   */
  /*          C  AAAA   */
  /*          C         */
  PORTA &= 0b11111100;  /* Port A (STBN_A = 1, BGE_A = 1)*/
  PORTA |= 0b00001100;  /* Port A */ 
  DDRA  |= 0b00001100;  /* Port A (STBN_A, BGE_A - output)*/  
  DDRA  &= 0b11111100;  /* Port A (ERRN_A, RxEN_A - input)*/ 
  
  /*               SR   */
  /*               TX   */
  /*               BE   */
  /*               NN   */
  /*               BB   */
  PORTB |= 0b00000010;  /* Port B (STBN_B = 1)*/
  PORTB &= 0b11111110;  /* Port B */
  DDRB  |= 0b00000010;  /* Port B STBN_B - output */ 
  DDRB  &= 0b11111110;  /* Port B RXEN_B - input  */ 
  
  /*         XMMEEBIX   */
  /*         COOCRGRI   */
  /*         LDDLREQR   */
  /*         KBAKNB Q   */
  /*         S   B      */
  PORTE |= 0b00000100;  /* Port E (BGE_B = 1)*/
  PORTE &= 0b11110111;  /* Port E */
  DDRE  |= 0b00000100;  /* Port E (BGE_B as output) */ 
  DDRE  &= 0b11110111;  /* Port E (ERRN_B as input) */ 
  
  /*        ETTRETTR */
  /*        NXXXNXXX */
  /*        |EDD|EDD */
  /*        |N|||N|| */
  /*        BBBBAAAA */
  PTH   = 0b11111111;  /* Port H */ 
  DDRH  = 0b11101110;  /* Port H */ 
  
  /* FR CC STB1..4 outputs */
  /*         SSSS    */
  /*         TTTT    */
  /*         BBBB    */
  /*         3210    */  
  PTJ  &= 0b10000111;  /* Port J */ 
  DDRJ |= 0b01111000;  /* Port J */ 
}






/************************************************************************************/

