/*******************************************************************************
 * ON Semiconductor                                                            *
 * NCV7381B Evaluation Board V1                                                *
 *                                                                             *
 * Cpu.c                                                                       *
 * S12XF512 CPU initialization                                                 *
 * Interrupts configuration                                                    *
 *                                                                             *
 * Created by: Filip Brtan                                                     *
 * Date: 20.06.2019                                                            *
 *******************************************************************************/

/* MODULE Cpu. */

/** S12X derivative information */ 
#include "mc9s12xf512.h"  

/** Common defines and macros */
#include <hidef.h>   

/** Variable types and common definitions */
#include "typedefs.h"   

/** XGATE definitions */
#include "xgate.h"

#pragma DATA_SEG DEFAULT
#pragma CODE_SEG DEFAULT

/* Global variables */
volatile byte CCR_reg;                 /* Current CCR reegister */

/*******************************************************************************
*
* brief    PLL initialization, Bus Clock @ 40MHz
* param    void
* return   void
*/
void Clock_Settings(void)
{    
    /*  System clock initialization */
    /* OSCCLK = 8MHz */
    
    /* CLKSEL: PLLSEL=0,PSTP=0,XCLKS=0,??=0,PLLWAI=0,??=0,RTIWAI=1,COPWAI=0 */
    CLKSEL = 0;              /* Select clock source from XTAL               */ 
    PLLCTL_PLLON = 0;        /* Disable the PLL                             */
    //PLLCTL_PRE = 0;          /* RTI stops running during Pseudo Stop Mode   */
    //CLKSEL_RTIWAI = 1;       /* RTI stops in Wait Mode                      */
              
    /* Fref = OSCCLK/(REFDIV+1) = 8MHz/(0+1) = 8MHz -> REFFRQ= 0b10         */    
    REFDV_REFDIV = 0x00;     /* PLL divider register                        */ 
    REFDV_REFFRQ = 0b10;     /* Reference Clock Frequency Selection         */
    
    /* Fvco = 2*OSCCLK*(SYNR+1)/(REFDIV+1) = 2*8MHz* (4+1)/(0+1) = 80MHz    */
    /*                                     -> VCOFRQ = 0b01                 */        
    SYNR_SYNDIV = 0x04;      /* PLL synthesizer register                    */
    SYNR_VCOFRQ = 0b01;      /* VCO Clock Frequency Selection               */
    
    POSTDIV_POSTDIV = 0x00;  /* Post Divider Register                       */
    /* Fpll  = fvco  / (2*POSTDIV)                                          */
    /* 80MHz = 80MHz / (1)                                                  */                                                    
    /* If POSTDIV = 0x00 then fPLL is identical to fVCO (divide by one).    */    
    /* Fbus = fpll/2 = 80MHz/2 = 40MHz                                      */
                                                                          
    PLLCTL_PLLON = 1;        /* PLL enabled                                 */
    
    while(!CRGFLG_LOCK)      /* Wait until the PLL is within the desired tolerance of the target frequency */
                
    CLKSEL_PLLSEL = 1;       /* Select clock source from PLL                */      
}


/*******************************************************************************
*
* brief    FlexRay IPLL initialization @ 80MHz
* param    void
* return   void
*/
void IPLL_Startup(void)
{
    /* Fref = OSCCLK/(REFDIV+1) = 8MHz/1 = 8MHz -> REFFRQ = 0b10      */ 
    CGMREFDV_REFDIV = 0x00;    /* CGMIPLL Reference Divider Register  */
    CGMREFDV_REFFRQ = 0b10;    /* Reference Clock Frequency Selection */
    
    /* Fvco = 2*OSCCLK*(SYNR+1)/(REFDV+1) = 2*8MHz*(4+1)/1 = 80MHz -> VCOFRQ = 01b */
    CGMSYNR_SYNDIV = 0x04;         
    CGMSYNR_VCOFRQ = 0b01;

    CGMCTL_DIV2 = 0;     /* DIV2=0 -> IPLL = Fvco  */ 

    CGMCTL_PLLON = 1;    /* IPLL enabled */

    while(!CGMFLG_LOCK); /* Wait until the PLL is within the desired tolerance of the target frequency */
}

/*******************************************************************************
*
* brief    CPU, XGATE, memory and ports settings
* param    void
* return   void
*/
void Peripheral_Settings(void)
{  
  /*******************************************************************************/
  /* Common initialization of the CPU registers */
  /*******************************************************************************/
  
  /* FlexRay module access all memory */
  RAMFRL = 0x00;
  RAMFRU = 0xFF;
  
  /* COP module configuration */
  /* COPCTL: WCOP=0, RSBCK=0, WRTMASK=0, ??=0, ??=0, CR2=0, CR1=0, CR0=0, */
  //COPCTL = 0x00;  /* COP disabled */ 

  /* MMCCTL1: TGMRAMON=0,??=0,EEEIFRON=0,PGMIFRON=0,RAMHM=0,EROMON=0,ROMHM=0,ROMON=1 */
  MMCCTL1 = 1;    /* Enables Flash in the memory map */  

  /* DIRECT: DP15=0,DP14=0,DP13=0,DP12=0,DP11=0,DP10=0,DP9=0,DP8=0 */                
  DIRECT = 0;     /* Default location for direct page */ 

  /* IVBR: IVB_ADDR=255 */             
  IVBR = 0xFF;    /* Default vector base address */    

  /* ECLKCTL: NECLK=0,NCLKX2=1,DIV16=0,EDIV4=0,EDIV3=0,EDIV2=0,EDIV1=0,EDIV0=0 */
  ECLKCTL = 0x40;   /* Disable free-running clock on the ECLK and ECLKX2 pin. */                                                

  /*******************************************************************************/
  /* Voltage regulator settings*/
  /* High Temperature Control Register */
  VREGHTCL = 0x10;        
  /* Control Register VREGCTRL */
  VREGCTRL = 0x00;             
  /* Autonomous Periodical Interrupt Control Register */
  VREGAPICL = 0x00;       
  /*Autonomous Periodical Interrupt Rate High and Low Register */
  VREGAPIRH = 0x00;  
  VREGAPIRL = 0x00;
    
  /*******************************************************************************/
  /* Ports configuration */
  /*******************************************************************************/
    
  /*** PORT AD ***/
  /* ATD Input Enable Register */
  ATDDIENH = 0x00;  
  ATDDIENL = 0x00;  /* Disable digital input buffer on PAD pins */           
  
  /* Port AD Data Register */
  PT0AD = 0b00000000;  /* PT0AD[15:8] */
  PT1AD = 0b00000000;  /* PT1AD[ 7:0] Initial output value = 0 (LEDs OFF) */    
  
  /* Port AD Pull Up Enable Register */
  PER0AD = 0x00; /* PER0AD[15:8] */             
  PER1AD = 0x00; /* PER1AD[ 7:0] (Pull-up devices disabled) */           
  
  /* Port AD Data Direction Register */
  DDR0AD = 0b00000000; /* DDR0AD[15:8] */
  DDR1AD = 0b00001000; /* DDR1AD[ 7:0] (pin 3 configured as output - mcu_INH_5V_bd) */
  
  /* Port AD Reduced Drive Register */
  RDR0AD = 0x00; /* RDR0AD[15:0] */
  RDR1AD = 0x00; /* RDR0AD[ 7:0] (Full drive strength enabled) */
  

  /* PORTx  Port x Data Register */;
  /* DDRx   Port x Data Direction Register (0-input / 1-output) */
  
  /************************/
  /* Port A               */
  /************************/
  /* PIN    | Dir | Level */
  /*--------|-----|-------*/
  /* INHVCC | OUT | 0     */
  /* STBN_A | OUT | 0     */
  /* BGE_A  | OUT | 0     */
  /* ERRN_A | IN  | x     */
  /* RxEN_A | IN  | x     */
  /************************/
  /*         I  BSER    */
  /*         N  GTRX    */
  /*         H  EBRE    */
  /*         V   NNN    */
  /*         C  AAAA    */
  /*         C          */
  PORTA = 0b00000000;   /* Port A Data */
  DDRA  = 0b11111100;   /* Port A Direction */
 
  
  /************************/
  /* Port B               */
  /************************/
  /* PIN    | Dir | Level */
  /*--------|-----|-------*/
  /* STBN_B | OUT | 0     */
  /* RxEN_B | IN  | x     */
  /************************/
  /*              SR    */
  /*              TX    */
  /*              BE    */
  /*              NN    */
  /*              BB    */
  PORTB = 0b00000000;   /* Port B Data*/
  DDRB  = 0b00000010;   /* Port B Direction */ 
  
  PORTC = 0x00;         /* Port C */
  DDRC  = 0x00;         /* Not implemented */ 
  
  PORTD = 0b00000000;   /* Port D (NOT USED) */
  DDRD  = 0b00111111;   /* Port D as output */
  
  /************************/
  /* Port E               */
  /************************/
  /* PIN    | Dir | Level */
  /*--------|-----|-------*/
  /* BGE_B  | OUT | 0     */
  /* ERRN_B | IN  | x     */
  /************************/
  /*        XMMEEBIX    */
  /*        COOCRGRI    */
  /*        LDDLREQR    */
  /*        KBAKNB Q    */
  /*        S   B       */
  PORTE = 0b00000000;   /* Port E Data*/
  DDRE  = 0b00010100;   /* Port E Direction */ 
  
  PORTK = 0;            /* Port K */
  DDRK  = 0x00;         /* Not implemented */
  
  /* Pull-up Control Register for ports A, B, C, D, E, K.  */
  /*        KB-EDCBA */
  PUCR  = 0b00000000; 
   
  /* Reduced drive on ports A, B, C, D, E, K. */   
  /*        K--EDCBA */     
  RDRIV = 0b11111111; /* Reduced drive selected */
  
  
  /***************************************/
  /* Port H                              */
  /***************************************/
  /* PIN    | Dir | Level                */
  /*--------|-----|----------------------*/
  /* EN_B   | OUT | 0                    */
  /* TxEN_B | OUT | 1                    */
  /* TxD_B  | OUT | 1                    */
  /* RxD_B  | IN  | x                    */
  /* EN_A   | OUT | 0                    */
  /* TxEN_A | OUT | 1                    */
  /* TxD_A  | OUT | 1                    */
  /* RxD_A  | IN  | x                    */
  /***************************************/
  /*          ETTRETTR                   */
  /*          NXXXNXXX                   */
  /*          |EDD|EDD                   */
  /*          |N|||N||                   */
  /*          BBBBAAAA                   */
     PTH  = 0b01110111; /* Port H Data   */ 
     DDRH = 0b11101110; /* Port H Dir    */ 
  /***************************************/
  
  
  /***************************************/
  /* Port J                              */
  /* FR CC STROBE outputs                */
  /***************************************/
  /* PIN    | Dir | Level                */
  /*--------|-----|----------------------*/
  /* STB3   | OUT | 0                    */
  /* STB2   | OUT | 0                    */
  /* STB1   | OUT | 0                    */
  /* STB0   | OUT | 0                    */
  /***************************************/
  /*           SSSS                      */
  /*           TTTT                      */
  /*           BBBB                      */
  /*           3210                      */  
     PTJ  = 0b00000000;  /* Port J Data  */ 
     DDRJ = 0b11111111;  /* Port J Dir   */
  /***************************************/ 
  
  
  /***************************************/
  /* Port M                              */
  /* CAN bus driver signals (CANRX-input)*/
  /***************************************/
  /* PIN    | Dir | Level                */
  /*--------|-----|----------------------*/
  /* STB    | OUT | 1                    */
  /* TX     | OUT | 1                    */
  /* RX     | IN  | x                    */
  /***************************************/
  /*         S   TR                      */
  /*         T   XX                      */           
  /*         B   ||                      */
  PTM  = 0b00100011;  /* Port M Data     */
  DDRM = 0b11111110;  /* Port M Dir      */ 
  /***************************************/
  
  /* LED Outputs */
  PTP  = 0b11100000;  /* Port P */
  DDRP = 0b11111111;  /* Port P */ 
  
  
  /***************************************/
  /* Port S                              */
  /* UART0 pins and Aux outputs          */
  /***************************************/
  /* PIN    | Dir | Level                */
  /*--------|-----|----------------------*/
  /* SS     | OUT | 0                    */
  /* SCK    | OUT | 0                    */
  /* MOSI   | OUT | 0                    */
  /* MISO   | OUT | 0                    */
  /* TXD1   | OUT | 0                    */
  /* RXD1   | OUT | 0                    */
  /* TXD0   | OUT | 1                    */
  /* RXD0   | IN  | x                    */
  /***************************************/
  /*       SSMMTRTR                      */
  /*       SCOIXXXX                      */
  /*       |KSSDDDD                      */
  /*       ||IO1100                      */
  PTS  = 0b00000011;  /* Port S Data     */
  DDRS = 0b11111110;  /* Port S Dir      */ 
  /***************************************/
  
 
  /* Switch input */
  PTT  = 0b00000000;  /* Port T */ 
  DDRT = 0b00000000;  /* Port T */ 
  
  /* PTIx Port x Input Register */
  PTIH = 0x00; /* Port H */
  PTIJ = 0x00; /* Port J */
  PTIM = 0x00; /* Port M */
  PTIP = 0x00; /* Port P */
  PTIS = 0x00; /* Port S */
  PTIT = 0x00; /* Port T */
                         
  /* RDRx Port x Reduced Drive Register (0-Full drive strength / 1-Reduced drive) */
  RDRH = 0x00;  /* Port H */                     
  RDRJ = 0x00;  /* Port J */                        
  RDRM = 0x00;  /* Port M */                       
  RDRP = 0x00;  /* Port P */                       
  RDRS = 0x00;  /* Port S */                     
  RDRT = 0x00;  /* Port T */  
  
  /* PERx Port x Pull Device Enable Register (0-Pull device disabled / 1-Pull device enabled.)*/
  PERH = 0x00; /* Port H */
  PERJ = 0b00; /* Port J */
  PERM = 0x00; /* Port M */
  PERP = 0x00; /* Port P */
  PERS = 0x00; /* Port S */
  PERT = 0xFF; /* Port T */
  
  /* PPSx Port x Polarity Select Register (0-pull-up device / 1-pull-down device) */
  PPSH = 0x00; /* Port H */
  PPSJ = 0x00; /* Port J */
  PPSM = 0x00; /* Port M */
  PPSP = 0x00; /* Port P */
  PPSS = 0x00; /* Port S */
  PPST = 0x00; /* Port T */
  
  /* WOMx Port x Wired-Or Mode Register  */
  WOMM = 0x00; /* Port M */
  WOMS = 0x00; /* Port S */      
}


/*******************************************************************************/
/* Programmable Interrupt Timer (PIT) module registers configuration           */
/*******************************************************************************/
void PIT_Init(void)
{
  PITCFLMT_PITE = 0; 
  PITCFLMT_PITSWAI = 0;        
  PITTRIGSTAT = 128;        
  PITMTLD0 = 3;                 
  PITMTLD1 = 0;                 
  PITLD0 = 0;                  
  PITLD1 = 0;                
  PITMUX = 0;                   
  PITCE = 3;                    
  PITTF = 3;                    
  PITINTE = 0;                  
  PITCSTP = 3;                  
  PITTRIGE = 0;                 
  PITTRIGOUT = 0;               
  PITTRIGCTL = 0;               
  PITCFLMT_PITE = 1;              
}

/*******************************************************************************/
/* Software reset of MCU (Setting watchdog and waiting for overflow)           */
/*******************************************************************************/
void MCU_Software_Reset(void){
  // Set watchdog but do not set watchdog reset commands
  COPCTL = 0b00000001; 
}


/*******************************************************************************/
/* Interrupts configuration                                                    */
/*******************************************************************************/
void Interrupts_Init(void)
{
    /* Access to configuration data registers for interrupts */ 
    INT_CFADDR = 0xF0;  /* Vectors from 0xFFF0 to 0xFFF8 */  
    INT_CFDATA0 = 0x06; /* Real Time Interrupt, priority 6 */    
            
    TIOS_IOS7 = 0;      /* Channel 7 configured as input capture */    
    TCTL3_EDG7B = 1;    /* Channel 7 input capture enabled in falling edges */    
    TCTL3_EDG7A = 0;                             
    
    ICSYS = 0;          /* Input capture channels in normal mode */
    PACTL = 0;          /* Pulse acumulator disabled */                                      
                                
    RTICTL_RTDEC = 1;   /* Real time interrupt decimal divider */                                                                      
    RTICTL_RTR = 0x55;  /* Divider = 300x10^3; RTI period=300k/8MHz = 37.5 ms */  
                        
    TSCR2 = 4;          /* Set timer prescaler to 16 */
  
    TFLG1_C7F = 1;      /* Clear input capture 7 interrupt request flag */
    TIE_C7I = 1;        /* Interrupt enabled for Timer channel 7 */
        
    /* External IRQ Settings */
    IRQCR_IRQE = 1;     /* External IRQ responds only to falling edges */
    IRQCR_IRQEN = 1;    /* External IRQ is enabled */ 
    //XIRQ_ENABLE():      /* Enable XIRQ */  
    
    CRGFLG_RTIF = 1;    /* Clear real time interrupt flag */    
    CRGINT_RTIE = 1;    /* Enable real time interrupt */    
    TSCR1_TEN = 1;      /* Timer enable */
}

/*******************************************************************************/


