
/*******************************************************************************
 * ON Semiconductor                                                            *
 * NCV7381B Evaluation Board V1                                                *
 *                                                                             *
 * ADC.c                                                                       *
 * Voltage measurement routines                                                *
 *                                                                             *
 * Created by: Filip Brtan                                                     *
 * Date: 20.06.2019                                                            *
 *******************************************************************************/

/* S12X derivative information */ 
#include "mc9s12xf512.h" 
/* Variable types and common definitions */
#include "typedefs.h"  
/* ADC header file */
#include "ADC.h"
/* Interrupt service routines prototypes */ 
#include "Interrupts.h"

/*******************************************************************************/

#pragma MESSAGE DISABLE C5703          /* Disable warning C5703 "Parameter is not referenced" */
#pragma MESSAGE DISABLE C4002          /* Disable warning C4002 "Result not used is ignored" */
#pragma MESSAGE DISABLE C12056         /* Disable warning C12056 "SP debug info incorrect because of optimization or inline assembler" */

#pragma DATA_SEG ADC_DATA
#pragma CODE_SEG ADC_CODE
#pragma CONST_SEG ADC_CONST            /* Constant section for this module */

/* Modes definition */
#define ADC_STOP            0              /* STOP state           */
#define ADC_MEASURE         1              /* MESURE state         */
#define ADC_CONTINUOUS      2              /* CONTINUOUS state     */
#define ADC_SINGLE          3              /* SINGLE state         */

/* Global variables */
#define NO_OF_ADC_CHANNELS 6 
static const byte Channels[] = {0,1,2,8,9,10};     /* Contents for the device control register */
static bool Result_Ready;              /* Measurement finish flag */
static byte Meas_Sum;                  /* Number of measured channels */
volatile static byte ModeFlg;          /* Current state of device */
static word ADC_Results[NO_OF_ADC_CHANNELS];            /* Array of measured values */
/*******************************************************************************/

/* ADC Interrupt Service Routine */
#pragma CODE_SEG __NEAR_SEG NON_BANKED
void interrupt ADC_Conv_Compl_Isr(void)
{
  ADC_Results[Meas_Sum] = ATDDR0;      /* Save measured value */
  Meas_Sum++;                          /* Number of measurement */
  if (Meas_Sum >= NO_OF_ADC_CHANNELS) {                 /* Is number of measurement equal to the number of channels? */
    Result_Ready = TRUE;               /* Measured values are available */
    //ADC_OnEnd();                       /* Invoke user event */
    ModeFlg = ADC_STOP;                /* Set the device to the stop mode */
  } else {
    ATDCTL5 = Channels[Meas_Sum];      /* Start measurement of next channel */
  }
}
#pragma CODE_SEG ADC_CODE
/*******************************************************************************/

/* Perform one ADC measurement */
byte ADC_Measure(bool WaitForResult)
{
  if (ModeFlg != ADC_STOP) {           /* Is the device in different mode than "stop"? */
    return ERR_BUSY;                   /* If yes then error */
  }
  ModeFlg = ADC_MEASURE;               /* Set state of device to the measure mode */
  Result_Ready = FALSE;                /* Output values aren't available */
  Meas_Sum = 0;                        /* Set the number of measured channels to 0 */
  ATDCTL5 = 0;                         /* Start the conversion */
  if (WaitForResult) {                 /* Is WaitForResult TRUE? */
    while (ModeFlg == ADC_MEASURE) {}  /* If yes then wait for end of measurement */
  }
  return ERR_OK;                       /* OK */
}
/*******************************************************************************/

/* Get ADC measurement results from all active channels */
byte ADC_GetValue16(word *Values)
{
  if (!Result_Ready) {                 /* Is measured value(s) available? */
    return ERR_NOTAVAIL;               /* If no then error */
  }
  /* Note: Next 6 lines are speed optimized */
  *Values++ = ADC_Results[0]; /* Save measured values to the output buffer */
  *Values++ = ADC_Results[1]; /* Save measured values to the output buffer */
  *Values++ = ADC_Results[2]; /* Save measured values to the output buffer */
  *Values++ = ADC_Results[3]; /* Save measured values to the output buffer */
  *Values++ = ADC_Results[4]; /* Save measured values to the output buffer */
  *Values   = ADC_Results[5]; /* Save measured values to the output buffer */
  return ERR_OK;              /* OK */
}
/*******************************************************************************/

/* Get ADC measurement result from one specific channel */
byte ADC_GetChanValue16(byte Channel, word *Value)
{
  if (Channel >= NO_OF_ADC_CHANNELS) { /* Is channel number greater than or equal to 6 */
    return ERR_RANGE;                  /* If yes then error */
  }
  if (!Result_Ready) {                 /* Is measured value(s) available? */
    return ERR_NOTAVAIL;
  }
  *Value = ADC_Results[Channel];       /* Save measured values to the output buffer */
  return ERR_OK;                       /* OK */
}
/*******************************************************************************/

/* ADC module initialization */
void ADC_init(void)
{
  Result_Ready = FALSE;                /* No measured value */
  ModeFlg = ADC_STOP;                  /* Device isn't running */
  /* ATDCTL4: SMP2=1,SMP1=1,SMP0=1,PRS4=0,PRS3=1,PRS2=1,PRS1=1,PRS0=0 */
  ATDCTL4 = 238;                       /* Set sample time and prescaler */
  /* ATDCTL3: DJM=1,S8C=0,S4C=0,S2C=0,S1C=1,FIFO=0,FRZ1=0,FRZ0=0 */
  ATDCTL3 = 136;                       /* Set ATD control register 3 */
  /* ATDCTL0: ??=0,??=0,??=0,??=0,WRAP3=1,WRAP2=1,WRAP1=1,WRAP0=1 */
  ATDCTL0 = 15;                        /* Set wrap around */
  /* ATDCTL1: ETRIGSEL=0,SRES1=1,SRES0=0,SMP_DIS=1,ETRIGCH3=1,ETRIGCH2=1,ETRIGCH1=1,ETRIGCH0=1 */
  ATDCTL1 = 95;                        /* Set resolution and discharge */
  /* ATDCTL2: ??=0,AFFC=1,ICLKSTP=0,ETRIGLE=0,ETRIGP=0,ETRIGE=0,ASCIE=1,ACMPIE=0 */
  ATDCTL2 = 66;                        /* Set ATD control register 2 */
}
/*******************************************************************************/



#pragma CODE_SEG DEFAULT
/*******************************************************************************/



















